<?php
/**
 * COCAVIP - Formulário de Contato
 * Processa e envia emails do formulário de contato
 */

// Configurações de segurança
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Responder a requisições OPTIONS (CORS preflight)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Verificar se é uma requisição POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Método não permitido']);
    exit();
}

// Configurações de email
$config = [
    'to_email' => 'cooperativacocavip@gmail.com',
    'to_name' => 'COCAVIP - Cooperativa de Catadores',
    'from_email' => 'noreply@cocavip.com.br',
    'from_name' => 'Site COCAVIP',
    'subject_prefix' => '[SITE COCAVIP] ',
    'smtp_host' => 'localhost', // Configurar conforme servidor
    'smtp_port' => 587,
    'smtp_username' => '', // Configurar conforme servidor
    'smtp_password' => '', // Configurar conforme servidor
    'use_smtp' => false // Usar mail() por padrão
];

/**
 * Função para sanitizar dados de entrada
 */
function sanitizeInput($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    return $data;
}

/**
 * Função para validar email
 */
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * Função para validar telefone brasileiro
 */
function validatePhone($phone) {
    $phone = preg_replace('/\D/', '', $phone);
    return strlen($phone) >= 10 && strlen($phone) <= 11;
}

/**
 * Função para enviar email usando mail()
 */
function sendEmailNative($to, $subject, $message, $headers) {
    return mail($to, $subject, $message, $headers);
}

/**
 * Função para enviar email usando SMTP (PHPMailer seria ideal)
 */
function sendEmailSMTP($config, $to, $subject, $message) {
    // Implementar SMTP se necessário
    // Por enquanto, usar mail() nativo
    return false;
}

/**
 * Função para gerar template HTML do email
 */
function generateEmailTemplate($data) {
    $template = '
    <!DOCTYPE html>
    <html lang="pt-BR">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Contato - COCAVIP</title>
        <style>
            body {
                font-family: Arial, sans-serif;
                line-height: 1.6;
                color: #333;
                max-width: 600px;
                margin: 0 auto;
                padding: 20px;
            }
            .header {
                background: linear-gradient(135deg, #2E8B57, #32CD32);
                color: white;
                padding: 20px;
                text-align: center;
                border-radius: 10px 10px 0 0;
            }
            .content {
                background: #f8f9fa;
                padding: 30px;
                border-radius: 0 0 10px 10px;
            }
            .field {
                margin-bottom: 15px;
                padding: 10px;
                background: white;
                border-radius: 5px;
                border-left: 4px solid #2E8B57;
            }
            .field strong {
                color: #2E8B57;
                display: block;
                margin-bottom: 5px;
            }
            .footer {
                text-align: center;
                margin-top: 20px;
                padding: 20px;
                background: #2C3E50;
                color: white;
                border-radius: 10px;
                font-size: 14px;
            }
            .logo {
                max-width: 100px;
                margin-bottom: 10px;
            }
        </style>
    </head>
    <body>
        <div class="header">
            <h1>Nova Mensagem do Site</h1>
            <p>COCAVIP - Cooperativa de Catadores</p>
        </div>
        
        <div class="content">
            <div class="field">
                <strong>Nome:</strong>
                ' . htmlspecialchars($data['name']) . '
            </div>
            
            <div class="field">
                <strong>E-mail:</strong>
                ' . htmlspecialchars($data['email']) . '
            </div>
            
            ' . (!empty($data['phone']) ? '
            <div class="field">
                <strong>Telefone:</strong>
                ' . htmlspecialchars($data['phone']) . '
            </div>
            ' : '') . '
            
            <div class="field">
                <strong>Assunto:</strong>
                ' . htmlspecialchars($data['subject']) . '
            </div>
            
            <div class="field">
                <strong>Mensagem:</strong>
                ' . nl2br(htmlspecialchars($data['message'])) . '
            </div>
            
            <div class="field">
                <strong>Data/Hora:</strong>
                ' . date('d/m/Y H:i:s') . '
            </div>
            
            <div class="field">
                <strong>IP do Remetente:</strong>
                ' . $_SERVER['REMOTE_ADDR'] . '
            </div>
        </div>
        
        <div class="footer">
            <p>Esta mensagem foi enviada através do formulário de contato do site da COCAVIP.</p>
            <p>Para responder, utilize o e-mail: ' . htmlspecialchars($data['email']) . '</p>
        </div>
    </body>
    </html>';
    
    return $template;
}

/**
 * Função para log de atividades
 */
function logActivity($message, $data = []) {
    $logFile = __DIR__ . '/logs/contact_' . date('Y-m') . '.log';
    $logDir = dirname($logFile);
    
    // Criar diretório de logs se não existir
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $logEntry = [
        'timestamp' => date('Y-m-d H:i:s'),
        'ip' => $_SERVER['REMOTE_ADDR'],
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
        'message' => $message,
        'data' => $data
    ];
    
    file_put_contents($logFile, json_encode($logEntry) . "\n", FILE_APPEND | LOCK_EX);
}

/**
 * Função para verificar rate limiting (proteção contra spam)
 */
function checkRateLimit($ip, $limit = 5, $window = 3600) {
    $logFile = __DIR__ . '/logs/rate_limit.log';
    $logDir = dirname($logFile);
    
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $now = time();
    $attempts = [];
    
    // Ler tentativas existentes
    if (file_exists($logFile)) {
        $lines = file($logFile, FILE_IGNORE_NEW_LINES);
        foreach ($lines as $line) {
            $data = json_decode($line, true);
            if ($data && $data['ip'] === $ip && ($now - $data['timestamp']) < $window) {
                $attempts[] = $data;
            }
        }
    }
    
    // Verificar se excedeu o limite
    if (count($attempts) >= $limit) {
        return false;
    }
    
    // Registrar nova tentativa
    $newAttempt = [
        'ip' => $ip,
        'timestamp' => $now
    ];
    file_put_contents($logFile, json_encode($newAttempt) . "\n", FILE_APPEND | LOCK_EX);
    
    return true;
}

try {
    // Verificar rate limiting
    $clientIP = $_SERVER['REMOTE_ADDR'];
    if (!checkRateLimit($clientIP)) {
        http_response_code(429);
        echo json_encode([
            'success' => false,
            'message' => 'Muitas tentativas. Tente novamente em alguns minutos.'
        ]);
        exit();
    }
    
    // Coletar e sanitizar dados do formulário
    $name = sanitizeInput($_POST['name'] ?? '');
    $email = sanitizeInput($_POST['email'] ?? '');
    $phone = sanitizeInput($_POST['phone'] ?? '');
    $subject = sanitizeInput($_POST['subject'] ?? '');
    $message = sanitizeInput($_POST['message'] ?? '');
    
    // Validações
    $errors = [];
    
    if (empty($name)) {
        $errors[] = 'Nome é obrigatório';
    } elseif (strlen($name) < 2) {
        $errors[] = 'Nome deve ter pelo menos 2 caracteres';
    }
    
    if (empty($email)) {
        $errors[] = 'E-mail é obrigatório';
    } elseif (!validateEmail($email)) {
        $errors[] = 'E-mail inválido';
    }
    
    if (!empty($phone) && !validatePhone($phone)) {
        $errors[] = 'Telefone inválido';
    }
    
    if (empty($subject)) {
        $errors[] = 'Assunto é obrigatório';
    } elseif (strlen($subject) < 5) {
        $errors[] = 'Assunto deve ter pelo menos 5 caracteres';
    }
    
    if (empty($message)) {
        $errors[] = 'Mensagem é obrigatória';
    } elseif (strlen($message) < 10) {
        $errors[] = 'Mensagem deve ter pelo menos 10 caracteres';
    }
    
    // Se houver erros, retornar
    if (!empty($errors)) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Dados inválidos',
            'errors' => $errors
        ]);
        exit();
    }
    
    // Preparar dados para o email
    $emailData = [
        'name' => $name,
        'email' => $email,
        'phone' => $phone,
        'subject' => $subject,
        'message' => $message
    ];
    
    // Gerar conteúdo do email
    $emailSubject = $config['subject_prefix'] . $subject;
    $emailBody = generateEmailTemplate($emailData);
    
    // Headers do email
    $headers = [
        'MIME-Version: 1.0',
        'Content-type: text/html; charset=UTF-8',
        'From: ' . $config['from_name'] . ' <' . $config['from_email'] . '>',
        'Reply-To: ' . $name . ' <' . $email . '>',
        'X-Mailer: PHP/' . phpversion(),
        'X-Priority: 3',
        'X-MSMail-Priority: Normal'
    ];
    
    $headerString = implode("\r\n", $headers);
    
    // Tentar enviar email
    $emailSent = false;
    
    if ($config['use_smtp']) {
        $emailSent = sendEmailSMTP($config, $config['to_email'], $emailSubject, $emailBody);
    } else {
        $emailSent = sendEmailNative($config['to_email'], $emailSubject, $emailBody, $headerString);
    }
    
    if ($emailSent) {
        // Log de sucesso
        logActivity('Email enviado com sucesso', [
            'name' => $name,
            'email' => $email,
            'subject' => $subject
        ]);
        
        // Resposta de sucesso
        echo json_encode([
            'success' => true,
            'message' => 'Mensagem enviada com sucesso! Entraremos em contato em breve.'
        ]);
    } else {
        // Log de erro
        logActivity('Falha ao enviar email', [
            'name' => $name,
            'email' => $email,
            'subject' => $subject,
            'error' => 'mail() function failed'
        ]);
        
        // Resposta de erro
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Erro interno do servidor. Tente novamente mais tarde.'
        ]);
    }
    
} catch (Exception $e) {
    // Log de exceção
    logActivity('Exceção capturada', [
        'error' => $e->getMessage(),
        'file' => $e->getFile(),
        'line' => $e->getLine()
    ]);
    
    // Resposta de erro
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erro interno do servidor. Tente novamente mais tarde.'
    ]);
}
?>

