<?php
/**
 * COCAVIP - Configurações do Sistema
 * Arquivo de configuração principal
 */

// Configurações de ambiente
define('ENVIRONMENT', 'production'); // development, staging, production

// Configurações de banco de dados (se necessário no futuro)
define('DB_HOST', 'localhost');
define('DB_NAME', 'cocavip_db');
define('DB_USER', 'cocavip_user');
define('DB_PASS', 'senha_segura');
define('DB_CHARSET', 'utf8mb4');

// Configurações de email
define('MAIL_HOST', 'localhost');
define('MAIL_PORT', 587);
define('MAIL_USERNAME', '');
define('MAIL_PASSWORD', '');
define('MAIL_FROM_EMAIL', 'noreply@cocavip.com.br');
define('MAIL_FROM_NAME', 'COCAVIP - Site Oficial');
define('MAIL_TO_EMAIL', 'cooperativacocavip@gmail.com');
define('MAIL_TO_NAME', 'COCAVIP - Cooperativa');

// Configurações de segurança
define('CSRF_TOKEN_NAME', 'csrf_token');
define('SESSION_LIFETIME', 3600); // 1 hora
define('MAX_LOGIN_ATTEMPTS', 5);
define('LOGIN_LOCKOUT_TIME', 900); // 15 minutos

// Configurações de upload (se necessário)
define('UPLOAD_MAX_SIZE', 5 * 1024 * 1024); // 5MB
define('UPLOAD_ALLOWED_TYPES', ['jpg', 'jpeg', 'png', 'gif', 'pdf', 'doc', 'docx']);
define('UPLOAD_PATH', __DIR__ . '/../uploads/');

// Configurações de cache
define('CACHE_ENABLED', true);
define('CACHE_LIFETIME', 3600); // 1 hora

// Configurações de log
define('LOG_ENABLED', true);
define('LOG_LEVEL', 'INFO'); // DEBUG, INFO, WARNING, ERROR
define('LOG_PATH', __DIR__ . '/logs/');

// Configurações de rate limiting
define('RATE_LIMIT_ENABLED', true);
define('RATE_LIMIT_REQUESTS', 10);
define('RATE_LIMIT_WINDOW', 60); // segundos

// URLs e caminhos
define('BASE_URL', 'https://cocavip.com.br');
define('ADMIN_URL', BASE_URL . '/admin');
define('ASSETS_URL', BASE_URL . '/assets');

// Configurações de SEO
define('SITE_NAME', 'COCAVIP - Cooperativa de Catadores');
define('SITE_DESCRIPTION', 'Cooperativa de Trabalho dos Catadores de Materiais Recicláveis Visão Pioneira de Icoaraci');
define('SITE_KEYWORDS', 'reciclagem, cooperativa, catadores, sustentabilidade, meio ambiente, Icoaraci, Belém, Pará');

// Configurações de redes sociais
define('FACEBOOK_URL', 'https://facebook.com/cooperativacocavip');
define('INSTAGRAM_URL', 'https://instagram.com/cocavip');
define('WHATSAPP_NUMBER', '5591991702977');
define('YOUTUBE_URL', '');

// Configurações de contato
define('CONTACT_EMAIL', 'cooperativacocavip@gmail.com');
define('CONTACT_PHONE', '(91) 99170-2977');
define('CONTACT_ADDRESS', 'Rua Oito de Maio, 268, Campina - Icoaraci - Belém-PA');

// Configurações de timezone
date_default_timezone_set('America/Belem');

// Configurações de erro baseadas no ambiente
if (ENVIRONMENT === 'development') {
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
    ini_set('log_errors', 1);
} else {
    error_reporting(0);
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
}

// Configurações de sessão
ini_set('session.cookie_httponly', 1);
ini_set('session.cookie_secure', 1);
ini_set('session.use_strict_mode', 1);

/**
 * Função para obter configuração
 */
function getConfig($key, $default = null) {
    return defined($key) ? constant($key) : $default;
}

/**
 * Função para verificar se está em desenvolvimento
 */
function isDevelopment() {
    return ENVIRONMENT === 'development';
}

/**
 * Função para verificar se está em produção
 */
function isProduction() {
    return ENVIRONMENT === 'production';
}

/**
 * Função para gerar URL absoluta
 */
function url($path = '') {
    return rtrim(BASE_URL, '/') . '/' . ltrim($path, '/');
}

/**
 * Função para gerar URL de assets
 */
function asset($path) {
    return rtrim(ASSETS_URL, '/') . '/' . ltrim($path, '/');
}

/**
 * Função para sanitizar dados
 */
function sanitize($data) {
    if (is_array($data)) {
        return array_map('sanitize', $data);
    }
    return htmlspecialchars(trim($data), ENT_QUOTES, 'UTF-8');
}

/**
 * Função para validar CSRF token
 */
function validateCSRF($token) {
    if (!isset($_SESSION[CSRF_TOKEN_NAME])) {
        return false;
    }
    return hash_equals($_SESSION[CSRF_TOKEN_NAME], $token);
}

/**
 * Função para gerar CSRF token
 */
function generateCSRF() {
    if (!isset($_SESSION[CSRF_TOKEN_NAME])) {
        $_SESSION[CSRF_TOKEN_NAME] = bin2hex(random_bytes(32));
    }
    return $_SESSION[CSRF_TOKEN_NAME];
}

/**
 * Função para log personalizado
 */
function writeLog($message, $level = 'INFO', $context = []) {
    if (!LOG_ENABLED) {
        return;
    }
    
    $logFile = LOG_PATH . 'app_' . date('Y-m-d') . '.log';
    $logDir = dirname($logFile);
    
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $logEntry = [
        'timestamp' => date('Y-m-d H:i:s'),
        'level' => $level,
        'message' => $message,
        'context' => $context,
        'ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
    ];
    
    file_put_contents($logFile, json_encode($logEntry) . "\n", FILE_APPEND | LOCK_EX);
}

/**
 * Função para resposta JSON
 */
function jsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * Função para verificar se é requisição AJAX
 */
function isAjax() {
    return isset($_SERVER['HTTP_X_REQUESTED_WITH']) && 
           strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
}

/**
 * Função para redirecionamento
 */
function redirect($url, $statusCode = 302) {
    header("Location: $url", true, $statusCode);
    exit;
}

/**
 * Autoloader simples para classes
 */
spl_autoload_register(function ($className) {
    $classFile = __DIR__ . '/classes/' . $className . '.php';
    if (file_exists($classFile)) {
        require_once $classFile;
    }
});

// Inicializar sessão se não estiver ativa
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
?>

